//
//  AD5272.c
//  dstat-firmware
//
//  Created by Michael Dryden on 2017-02-18.
//  Copyright © 2017 Michael Dryden. All rights reserved.
//

#include <twi_master.h>
#include <twi_common.h>

#include "ad5272.h"
#include "../config/dstat_config.h"

#define AD5272_TWI  EXT_TWI0
#define AD5272_ADDR 0b0101110

#define WRITE_MASK  0x3FFF // First two bits 0
#define CMD_MASK    0x3c00 // Next 4 bits
#define DATA_MASK   0x03FF // Last 10 bits
#define CTRL_MASK   0x0007 // Last 3 bits

// Operations
#define OP_WRITE_RDAC       0x0400
#define OP_READ_RDAC        0x0800
#define OP_BURN_RDAC        0x0C00
#define OP_RESET            0x1000
#define OP_READ_TP_ADDR     0x1400
#define OP_READ_TP_LAST     0x1800
#define OP_WRITE_CTRL       0x1C00
#define OP_READ_CTRL        0x2000
#define OP_SHUTDOWN         0x2400

// CTRL register bitmasks
#define CTRL_50TP       1
#define CTRL_RDAC_WP    1 << 1
#define CTRL_PERFORM    1 << 2
#define CTRL_50TP_SUCC  1 << 3

static uint16_t read(uint16_t address);
static void write(uint16_t input);

static uint16_t read(uint16_t address) {
    /**
     * Reads out shift register
     *
     * @param address Address to read out
     * @return Contents of shift register
     */
    twi_package_t twi_pack;
    
    uint16_t buffer;
    
    union{
        uint8_t uint8[2];
        uint16_t uint16;
    } input_buffer;
    
    input_buffer.uint16 = address;
    
    twi_pack.addr[0] = input_buffer.uint8[1];
    twi_pack.addr[1] = input_buffer.uint8[0];
    twi_pack.addr_length = 2;
    twi_pack.buffer = &buffer;
    twi_pack.chip = AD5272_ADDR;
    twi_pack.no_wait = false;
    twi_pack.length = 2;
    
    twi_master_read(&AD5272_TWI, &twi_pack);
    
    return swap16(buffer);
}

static void write(uint16_t input) {
    /**
     * Writes to shift register
     *
     * @param input Contents of shift register
     */
    twi_package_t twi_pack;
    
    uint16_t swapped = swap16(input);
    
    twi_pack.addr[0] = 0;
    twi_pack.addr_length = 0;
    twi_pack.buffer = &swapped;
    twi_pack.chip = AD5272_ADDR;
    twi_pack.no_wait = false;
    twi_pack.length = 2;
    
    twi_master_write(&AD5272_TWI, &twi_pack);
    if(twi_master_write(&AD5272_TWI, &twi_pack) != TWI_SUCCESS)
        printf("#ERR: AD5272 TWI write fail\r\n");
    
    return;
}

void ad5272_init(void) {
    /**
     * Enables RDAC writes
     */
    
    write(OP_WRITE_CTRL | CTRL_RDAC_WP);
    printf("#INFO: AD5272: RDAC writes enabled\r\n");
}

void ad5272_write_rdac(uint16_t input) {
    /**
     * Writes to shift register
     *
     * @param input Contents of shift register
     */
    input = OP_WRITE_RDAC | (input & DATA_MASK);
    write(input);
//    printf("#DBG RDAC: %u\r\n", read(OP_READ_RDAC) & DATA_MASK);
}
