//
//  eis.c
//  dstat-firmware
//
//  Created by Michael Dryden on 2017-03-20.
//  Copyright © 2017 Michael Dryden. All rights reserved.
//

#include <delay.h>

#include "eis.h"
#include "eis_misc.h"
#include "ds4550.h"
#include "ad5272.h"
#include "max5443.h"

uint8_t eis_status = EIS_STATUS_DONE;

void eis_init(void){
    ad5272_init();
    ds4550_init();
}

void eis_measure_auto(uint32_t start, uint32_t increment, uint16_t n_increments,
                             uint16_t settle_cycles, uint8_t pga_gain, uint8_t output_range,
                             uint8_t iv_gain, uint16_t attenuation, uint16_t offset,
                             uint8_t reference, uint16_t init_ms, cint16_t* result_buffer){
    /**
     Runs eis experiment in auto mode and writes result to buffer.
     
     @param start Start frequency in DDS units
     @param increment Frequency increment in DDS units
     @param n_increments Number of increments
     @param settle_cycles Number of settling cycles
     @param pga_gain PGA gain setting
     @param output_range Excitation voltage setting
     @param iv_gain IV converter gain setting
     @param attenuation gain setting
     @param offset offset voltage in max5443 units
     @param reference Reference connection setting
     @param init_ms Delay before first measurement in ms
     @param result_buffer Buffer to store results
     */
    eis_set_mclk(0);
    ad5933_set_params(start, increment, n_increments, settle_cycles, pga_gain, output_range, init_ms);
    eis_set_ref(reference);
    eis_set_gain(iv_gain);
    ad5272_write_rdac(attenuation);
    max5443_set_voltage1(offset);
    
    delay_ms(300);
    
    ad5933_sweep(result_buffer, n_increments);
    
    return;
}

void eis_set_params(uint32_t start, uint32_t increment, uint16_t n_increments,
                     uint16_t settle_cycles, uint8_t pga_gain, uint8_t output_range,
                     uint8_t iv_gain, uint16_t attenuation, uint16_t offset,
                     uint8_t reference, uint16_t mclk_div2, uint16_t init_ms){
    eis_set_mclk(mclk_div2);
    ad5933_set_params(start, increment, n_increments, settle_cycles, pga_gain, output_range, init_ms);
    eis_set_ref(reference);
    eis_set_gain(iv_gain);
    ad5272_write_rdac(attenuation);
    max5443_set_voltage1(offset);
    
    eis_status = EIS_STATUS_READY;
}

cint16_t eis_measure_one(void){
    cint16_t result;
    
    switch (eis_status) {
        case EIS_STATUS_READY:
            eis_status = EIS_STATUS_MEASURE;
            result = ad5933_measure_one(AD5933_MEASURE_FIRST);
            break;
        case EIS_STATUS_MEASURE:
            result = ad5933_measure_one(AD5933_MEASURE_NEXT);
            break;
        case EIS_STATUS_REPEAT:
            result = ad5933_measure_one(AD5933_MEASURE_REPEAT);
            break;
        default:
            result = ad5933_measure_one(AD5933_MEASURE_NEXT);
            break;
    }
    
    if (result.real == 0 && result.imag == 0){
        eis_status = EIS_STATUS_DONE;
    }
    
    return result;
}
