//
//  eis_misc.c
//  dstat-firmware
//
//  Created by Michael Dryden on 2017-03-02.
//  Copyright © 2017 Michael Dryden. All rights reserved.
//

#include "eis_misc.h"
#include "ds4550.h"
#include "ad5933.h"
#include <tc.h>
#include <ioport.h>
#include <delay.h>

void eis_set_gain(uint8_t gain){
    /**
     * Set EIS IV converter gain.
     *
     * @param gain Gain setting defined in eis_misc.h
     */
    
    delay_ms(30);
    switch (gain) {
        case EIS_IV_GAIN_0:
            ds4550_set_pins(0b110000000, 0b111000000);
            printf("#INFO: EIS gain 0\r\n");
            break;
        case EIS_IV_GAIN_100:
            ds4550_set_pins(0b111000000, 0b111000000);
            printf("#INFO: EIS gain 100\r\n");
            break;
        case EIS_IV_GAIN_3k:
            ds4550_set_pins(0b101000000, 0b111000000);
            printf("#INFO: EIS gain 3k\r\n");
            break;
        case EIS_IV_GAIN_30k:
            ds4550_set_pins(0b100000000, 0b111000000);
            printf("#INFO: EIS gain 30k\r\n");
            break;
        case EIS_IV_GAIN_300k:
            ds4550_set_pins(0b011000000, 0b111000000);
            printf("#INFO: EIS gain 300k\r\n");
            break;
        case EIS_IV_GAIN_3M:
            ds4550_set_pins(0b010000000, 0b111000000);
            printf("#INFO: EIS gain 3M\r\n");
            break;
        case EIS_IV_GAIN_30M:
            ds4550_set_pins(0b001000000, 0b111000000);
            printf("#INFO: EIS gain 30M\r\n");
            break;
        case EIS_IV_GAIN_100M:
            ds4550_set_pins(0b000000000, 0b111000000);
            printf("#INFO: EIS gain 100M\r\n");
            break;
        default:
            printf("#WAR: EIS: No gain provided\r\n");
            break;
    }
    
    return;
}

void eis_set_ref(uint8_t ref){
    /**
     * Set EIS reference resistor switches.
     *
     * @param ref Reference setting defined in eis_misc.h
     */
    
    delay_ms(30);
    switch (ref) {
        case EIS_REF_CELL:
            ds4550_set_pins(0b000001101, 0b000111111);
            printf("#INFO: EIS REF RE\r\n");
            break;
        case EIS_REF_R1:
            ds4550_set_pins(0b000010110, 0b000111111);
            printf("#INFO: EIS REF R1\r\n");
            break;
        case EIS_REF_R2:
            ds4550_set_pins(0b000100110, 0b000111111);
            printf("#INFO: EIS REF R2\r\n");
            break;
        case EIS_REF_PARALLEL:
            ds4550_set_pins(0b000110110, 0b000111111);
            printf("#INFO: EIS REF R1+2\r\n");
            break;
            
        default:
            printf("#WAR: EIS: No reference set.\r\n");
            break;
    }

    return;
}


void eis_set_mclk(uint16_t div_2){
    /**
     * Set EIS mclk frequency. Send 0 to reset to internal MCLK (16.776 MHz)
     *
     * @param div_2 half of the desired frequency divider (f = 1/(2*div_2))
     */
    if (div_2 == 0)
        ad5933_set_internal_mclk(1);
    else {
        ioport_set_pin_dir(IOPORT_CREATE_PIN(PORTC,2), IOPORT_DIR_OUTPUT);
        tc_enable(&TCC0);
        tc_set_wgm(&TCC0, TC_WG_FRQ);
        tc_write_cc(&TCC0, TC_CCA, div_2-1);
        tc_enable_cc_channels(&TCC0,TC_CCCEN);
        tc_write_clock_source(&TCC0, TC_CLKSEL_DIV1_gc);
        ad5933_set_internal_mclk(0);
    }
}
