/**
 * \file
 *
 * \brief Empty user application template
 *
 */

/*
 * Include header files for all drivers that have been imported from
 * Atmel Software Framework (ASF).
 */

#define __STDC_LIMIT_MACROS
#define __STDC_CONSTANT_MACROS

extern "C"{
#include <asf.h>
#include <string.h>
#include <math.h>
#include <stdint.h>
#include "ads1255.h"
#include "max5443.h"
}

//#include "new.cpp"
#include "cstddef"
#include "main.h"
#include "iterator"
#include "vector"
#include "queue"

//volatile uint8_t gain = POT_GAIN_30k;
volatile int32_t voltage = 0;
volatile uint16_t dacindex = 0;
uint16_t dacindex_stop = 0;
volatile int8_t up = 1;
volatile uint16_t iter = 0;
uint16_t* eis_ptr = 0;
volatile uint16_t cycles = 0;
volatile uint16_t samples = 0;
volatile uint16_t tcf0period = 0;
uint32_t skip_samples = 0;

uint8_t ads_buff = ADS_BUFF_ON;
uint8_t ads_datarate = ADS_DR_60;
uint8_t ads_pga = ADS_PGA_2;

int16_t eis_data[500];

typedef void (*port_callback_t) (void);

static port_callback_t portd_int0_callback;
static port_callback_t portd_int1_callback;

static void tcf0_ovf_callback(void);
static void tcf0_ovf_callback_eis(void);
static void tce0_ovf_callback_eis(void);
static void tcd0_ovf_callback_eis(void);
static void porte_int0_lsv(void);
static void tce1_ovf_callback_lsv(void);
static void lsv_cca_callback(void);
static void porte_int0_ca(void);
static void ca_cca_callback(void);

enum potgain_t {POT_GAIN_100, POT_GAIN_300, POT_GAIN_3k, POT_GAIN_30k, POT_GAIN_300k, POT_GAIN_3M, POT_GAIN_30M, POT_GAIN_500M};

void pot_set_gain(/*uint8_t*/ potgain_t gain);

potgain_t gain = POT_GAIN_3k;

inline potgain_t& operator++(potgain_t& x){
	const int i = static_cast<int>(x);
 	x = static_cast<potgain_t>(i + 1);
	return x;
 }
 
  inline potgain_t& operator--(potgain_t& x){
	const int i = static_cast<int>(x);
 	x = static_cast<potgain_t>(i - 1);
	return x;
 }

/*static void tce1_ovf_callback(void);*/

int8_t autogainswitch(void){
	extern int8_t over_under[ADS_OVER_UNDER_SAMPLES];
	int8_t overcurrent = 0;
	static uint8_t hysteresis = 0;
	static uint8_t last_return = 0;
	
	if (last_return==1)
	{
		last_return=0;
		return 1;
	}
	
	if (gain == POT_GAIN_500M)
		return 0;
	
	if (hysteresis < ADS_OVER_UNDER_SAMPLES-1){
		++hysteresis;
		return 0;
	}
	
	
	for (uint16_t i = 0; i < ADS_OVER_UNDER_SAMPLES; ++i)
	{
		overcurrent += over_under[i];
	}
	/*printf("OC: %i\n\r", overcurrent);*/
		
	if (overcurrent == ADS_OVER_UNDER_SAMPLES && gain > POT_GAIN_300)
	{
		pot_set_gain(--gain);
		last_return=1;
		hysteresis = 0;
		for (uint16_t i = 0; i < ADS_OVER_UNDER_SAMPLES; ++i)
			over_under[i]=0;
		
		return 1;
	}
	 
	if ((overcurrent*-1) == (ADS_OVER_UNDER_SAMPLES) && gain < POT_GAIN_30M)
	{
		if (hysteresis < ADS_OVER_UNDER_SAMPLES+3){
			++hysteresis;
			return 0;
		
		}
		pot_set_gain(++gain);
		last_return=1;
		hysteresis = 0;
		for (uint16_t i = 0; i < ADS_OVER_UNDER_SAMPLES; ++i)
			over_under[i]=0;
			
		return 1;
	}
	
	hysteresis = 0;
	return 0;
}

void pot_set_gain(/*uint8_t*/ potgain_t gain){
	switch (gain){
// 		case POT_GAIN_30k:
// 			ioport_configure_port_pin(&PORTD, PIN3_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
// 			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
// 			printf("INFO: 30k\n\r");
// 			break;
// 		case POT_GAIN_300k:
// 			ioport_configure_port_pin(&PORTD, PIN3_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
// 			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
// 			printf("INFO: 300k\n\r");
// 			break;
// 		case POT_GAIN_3M:
// 			ioport_configure_port_pin(&PORTD, PIN3_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
// 			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
// 			printf("INFO: 3M\n\r");
// 			break;
// 		case POT_GAIN_500M:
// 			ioport_configure_port_pin(&PORTD, PIN3_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
// 			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
// 			printf("INFO: 500M\n\r");;
// 			break;
		case POT_GAIN_500M:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT); //A
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT); //B
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT); //C
			printf("INFO: 500M\n\r");
			break;
		case POT_GAIN_30M:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			printf("INFO: 30M\n\r");
			break;
		case POT_GAIN_3M:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			printf("INFO: 3M\n\r");
			break;
		case POT_GAIN_300k:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			printf("INFO: 300k\n\r");
			break;
		case POT_GAIN_30k:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			printf("INFO: 30k\n\r");
			break;
		case POT_GAIN_3k:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			printf("INFO: 3k\n\r");
			break;
		case POT_GAIN_300:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			printf("INFO: 300\n\r");
			break;
		case POT_GAIN_100:
			ioport_configure_port_pin(&PORTB, PIN6_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTB, PIN7_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			ioport_configure_port_pin(&PORTD, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
			printf("INFO: 100\n\r");
			break;
			
		default:
			printf("WAR: Invalid pot gain.\n\r");
			break;
			
		return;
	}
}

inline void pot_exp_start(void){
	ioport_configure_port_pin(&PORTB, PIN4_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
	ioport_configure_port_pin(&PORTB, PIN5_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
	ioport_configure_port_pin(&PORTB, PIN3_bm, IOPORT_INIT_HIGH | IOPORT_DIR_OUTPUT);
}

inline void pot_exp_stop(void){
	ioport_configure_port_pin(&PORTB, PIN4_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
	ioport_configure_port_pin(&PORTB, PIN5_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
	ioport_configure_port_pin(&PORTB, PIN3_bm, IOPORT_INIT_LOW | IOPORT_DIR_OUTPUT);
}

void cv_experiment(int16_t v1, int16_t v2, int16_t start, uint8_t scans, uint16_t slope){
	// check if start is [v1,v2]
	int8_t firstrun = 1;
	
	if((start < v1 && start < v2) || (start > v1 && start > v2))
	{
		printf("ERR: Start must be within [v1, v2]\n\r");
		return;
	}
	
	RTC.CTRL = RTC_PRESCALER_OFF_gc;
	while (RTC.STATUS & RTC_SYNCBUSY_bm);
	RTC.CNT = 0;
	RTC.PER = 0xffff;
	RTC.CTRL = RTC_PRESCALER_DIV1024_gc; // 1 s tick
	
	
	while(scans > 0){
		if (start != v1){
			lsv_experiment(start,v1,slope,firstrun);
			firstrun = 0;
		}
		if (start == v2 && scans == 1)
			firstrun = -1;
		lsv_experiment(v1,v2,slope, firstrun);
		if (scans == 1)
			firstrun = -1;
		if (start != v2)
			lsv_experiment(v2,start,slope, firstrun);
		scans--;
		firstrun = 0;
	}
	
	printf("Time: %lu s \n\r", rtc_get_time());
	
	return;
}

void lsv_experiment(int16_t start, int16_t stop, uint16_t slope, int8_t first_run){
	//check experiment limits
	if(start<-1500 || start>1500 ||start==stop|| stop<-1500 || stop>1500 || slope>7000)
	{
		printf("ERR: Experiment parameters outside limits\n\r");
		return;
	}
	
	uint16_t dacindex_start = ceil(start*(65535/(double)3000)+32767);
	dacindex_stop = ceil(stop*(65535/(double)3000)+32767);
//	uint16_t period;
	uint32_t timer_period;
	uint16_t temp_div;

	
	printf("Firstrun:%i\n\r", first_run);
	
	max5443_set_voltage1(dacindex_start);
	
	if (first_run == 1 || first_run == 2){
		
		pot_exp_start();
		ads1255_rdatac();
		tc_enable(&TCC1);
		
		ads1255_sync();
		
//		voltage = TCE1.CNT;
		tc_enable(&TCC0);
		tc_set_overflow_interrupt_callback(&TCC0, tcf0_ovf_callback);
		tc_set_overflow_interrupt_callback(&TCC1, tce1_ovf_callback_lsv);
		tc_set_cca_interrupt_callback(&TCC1, lsv_cca_callback);
		portd_int0_callback = porte_int0_lsv;

		//set EVCH0 event
		EVSYS.CH0MUX = EVSYS_CHMUX_TCC0_OVF_gc;
		EVSYS.CH0CTRL = 0;

		timer_period = ceil(1/((double)slope/(3000./65535))*(F_CPU));
		temp_div = ceil(timer_period/65535.);
	
		if (temp_div <= 1){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV1_gc);
		}
		else if (temp_div == 2){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV2_gc);
			timer_period /= 2;
		}		
		else if (temp_div <= 4){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV4_gc);
			timer_period /= 4;
		}
		else if (temp_div <= 8){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV8_gc);
			timer_period /= 8;
		}
		else if (temp_div <= 64){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV64_gc);
			timer_period /= 64;
		}
		else if (temp_div <= 256){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV256_gc);
			timer_period /= 256;
		}
		else if (temp_div <= 1024){
			tc_write_clock_source(&TCC0,TC_CLKSEL_DIV1024_gc);
			timer_period /= 1024;
		}
		else{
			printf("ERR: Frequency/ADC rate is too low\n\r");
			return;
		}
	
		printf("Period:%lu\n\r", timer_period);
		ads1255_wakeup();
		tc_write_period(&TCC1, 0xffff);
		tc_write_period(&TCC0, (uint16_t)timer_period);
	}
	
	TCC1.CNT = dacindex_start;
	
	if (stop > start)
	{
		up = 1;
		tc_set_direction(&TCC1, TC_UP);
	}
	else
	{
		up = -1;
		tc_set_direction(&TCC1, TC_DOWN); 
	}	
	
	tc_write_cc(&TCC1, TC_CCA, dacindex_stop);
	tc_enable_cc_channels(&TCC1, TC_CCAEN);
	TCC0.CNT = 0;
	
	tc_set_cca_interrupt_level(&TCC1, TC_INT_LVL_HI);
	tc_set_overflow_interrupt_level(&TCC0, (TC_INT_LEVEL_t) TC_OVFINTLVL_MED_gc);
//	PORTE.INTCTRL = PORT_INT0LVL_LO_gc;
	PORTD.INTCTRL = PORT_INT0LVL_LO_gc;
	
	tc_write_clock_source(&TCC1, TC_CLKSEL_EVCH0_gc);
	
	
	while (up != 0)
	{	
	}	
	
	if (first_run == -1 || first_run == 2)
	{
		tc_disable(&TCC0);
		TCC0.CNT = 0x0;
		tc_disable(&TCC1);
		pot_exp_stop();
		ads1255_standby();
	}	
	
	return;
}

static void porte_int0_lsv(void){	
	if (autogainswitch()==0){
//		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTD, 5)));
		if (gain == POT_GAIN_300)
			printf("%u, %ld00000\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_3k)
			printf("%u, %ld0000\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_30k)
			printf("%u, %ld000\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_300k)
			printf("%u, %ld00\n\r", TCC1.CNT, ads1255_read());			
		else if (gain == POT_GAIN_3M)
			printf("%u, %ld0\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_30M)
			printf("%u, %ld\n\r", TCC1.CNT, ads1255_read());		
		else
			printf("%u, %lu\n\r", TCC1.CNT, ads1255_read());
	}
	else{
//		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTD, 5)));
		ads1255_read();
	}
}

static void tcf0_ovf_callback(void){
	max5443_set_voltage1(TCC1.CNT);
}

static void tce1_ovf_callback_lsv(void){
//	printf("TCE1_OVF\n\r");
//	PORTE.INTCTRL = PORT_INT0LVL_OFF_gc;
	PORTD.INTCTRL = PORT_INT0LVL_OFF_gc;
	tc_set_overflow_interrupt_level(&TCC0, (TC_INT_LEVEL_t)TC_OVFINTLVL_OFF_gc);
	tc_set_overflow_interrupt_level(&TCC1, (TC_INT_LEVEL_t)TC_OVFINTLVL_OFF_gc);
	up = 0;
	return;
}

static void lsv_cca_callback(void)
 {
//	printf("CCA\n\r");
//	PORTE.INTCTRL = PORT_INT0LVL_OFF_gc;
	PORTD.INTCTRL = PORT_INT0LVL_OFF_gc;
	tc_set_overflow_interrupt_level(&TCC0, (TC_INT_LEVEL_t)TC_OVFINTLVL_OFF_gc);
	tc_set_cca_interrupt_level(&TCC1, TC_INT_LVL_OFF);
	up = 0;
	return;
 }


void chronoamp(uint16_t steps, uint16_t step_dac[], uint16_t step_seconds[]){

	while (RTC.STATUS & RTC_SYNCBUSY_bm);
	RTC.PER = 999;
	while (RTC.STATUS & RTC_SYNCBUSY_bm);
	RTC.CTRL = RTC_PRESCALER_DIV1_gc;
	
	EVSYS.CH0MUX = EVSYS_CHMUX_RTC_OVF_gc;
	
	portd_int0_callback = porte_int0_ca;
	
	tc_enable(&TCC0);
	tc_set_cca_interrupt_callback(&TCC0, ca_cca_callback);
	
	ads1255_rdatac();
	ads1255_wakeup();
	
	tc_write_period(&TCC0,0xffff);
	tc_write_clock_source(&TCC0, TC_CLKSEL_EVCH0_gc);
	tc_set_direction(&TCC0, TC_UP);
	tc_enable_cc_channels(&TCC0, TC_CCAEN);
	tc_set_cca_interrupt_level(&TCC0, TC_INT_LVL_HI);
	TCC0.CNT = 0;
	
	pot_exp_start();
	
	for (uint8_t i = 0; i < steps; ++i)
	{	
		up = 1;
		tc_write_cc(&TCC0, TC_CCA, TCC0.CNT+step_seconds[i]-1);
		RTC.CNT=0;
		max5443_set_voltage1(step_dac[i]);
		//PORTE.INTCTRL = PORT_INT0LVL_LO_gc;
		PORTD.INTCTRL = PORT_INT0LVL_LO_gc;
		while (up !=0);
	}
	
	tc_set_cca_interrupt_level(&TCC0, TC_INT_LVL_OFF);
	tc_write_clock_source(&TCC0, TC_CLKSEL_OFF_gc);
	tc_disable(&TCC0);
	pot_exp_stop();
	ads1255_standby();

	return;
}

static void porte_int0_ca(void){	
	if (autogainswitch()==0){
		//while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTD, 5)));
		if (gain == POT_GAIN_300)
			printf("%u, %ld00000\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_3k)
			printf("%u, %ld0000\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_30k)
			printf("%u, %ld000\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_300k)
			printf("%u, %ld00\n\r", TCC1.CNT, ads1255_read());			
		else if (gain == POT_GAIN_3M)
			printf("%u, %ld0\n\r", TCC1.CNT, ads1255_read());
		else if (gain == POT_GAIN_30M)
			printf("%u, %ld\n\r", TCC1.CNT, ads1255_read());		
		else
			printf("%u, %lu\n\r", TCC1.CNT, ads1255_read());
	}
	else{
		//while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTD, 5)));
		ads1255_read();
	}
}

static void ca_cca_callback(void)
 {
	//PORTE.INTCTRL = PORT_INT0LVL_OFF_gc;
	PORTD.INTCTRL = PORT_INT0LVL_OFF_gc;
	up = 0;
	return;
 }

void swv_experiment(int16_t start, int16_t stop, uint16_t step, uint16_t pulse_height, uint16_t frequency){
	//check experiment limits 
// 	if((start-pulse_height)<-1500 || (start+pulse_height)>(1500)|| start == stop || stop<-1500 || (stop+pulse_height)>1500 || step<1 || step>1500 || pulse_height > 1500 || pulse_height < 1)
// 	{
// 		printf("ERR: Experiment parameters outside limits\n\r");
// 		//return;
// 	}

	int32_t forward = 0;
	int32_t reverse = 0;
	int32_t data_buffer;
	uint8_t direction;
	uint16_t dacindex_start = ceil((start)*(65535/(double)3000))+32767;
	uint16_t dacindex_stop = ceil(stop*(65535/(double)3000))+32767;
	uint16_t dacindex_step = ceil(step*(65535/(double)3000));
	uint16_t dacindex_pulse_height = ceil(pulse_height*(65535/(double)3000));
	uint16_t dacindex = dacindex_start;
	uint32_t period;
	int16_t fgain = -1;
	int16_t rgain = -1;

	
	if (start < stop)
		direction = 1;
	else
		direction = 0;
	
	tc_enable(&TCF0);
	
	frequency *= 2; //compensate for half-period triggers
	
	//calculate time to ADC trigger
	period = ceil((1/(double)frequency)*F_CPU);
	uint8_t temp_div = ceil((double)period/UINT16_MAX);
	
	if (temp_div == 1)
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV1_gc);
	else if (temp_div == 2){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV2_gc);
		period /= 2;
	}		
	else if (temp_div <= 4){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV4_gc);
		period /= 4;
	}
	else if (temp_div <= 8){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV8_gc);
		period /= 8;
	}
	else if (temp_div <= 64){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV64_gc);
		period /= 64;
	}
	else if (temp_div <= 256){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV256_gc);
		period /= 256;
	}
	else if (temp_div <= 1024){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV1024_gc);
		period /= 1024;
	}
	else{
		printf("Frequency/ADC rate is too low\n\r");
		return;
	}
	
	tc_write_period(&TCF0, (uint16_t)period);
	
	if (direction == 1)
		max5443_set_voltage1(dacindex+dacindex_pulse_height);
	else
		max5443_set_voltage1(dacindex-dacindex_pulse_height);
	
	ads1255_wakeup();
	ads1255_rdatac();
	ads1255_sync();
	
	pot_exp_start();
	TCF0.CNT = 0;
	while (!tc_is_overflow(&TCF0));
	ads1255_wakeup();
	TCF0.CNT = 0;
	
	while ((dacindex <= dacindex_stop && direction == 1) || (dacindex >= dacindex_stop && direction == 0)){
		tc_clear_overflow(&TCF0);
		
		if (fgain > -1)
			//printf("%u, %li, %li\n\r", dacindex, forward*fgain, reverse*rgain);
			printf("%u, %li%0*lu, %li%0*lu\n\r", dacindex, fgain, 0, forward, rgain, 0, reverse);
		
		forward = 0;
		reverse = 0;
		
		while (!tc_is_overflow(&TCF0)){
			autogainswitch();
			if (gain <= POT_GAIN_30M)
				fgain = 0;
			else
				fgain = gain-1;

			
			while (ioport_pin_is_low(IOPORT_CREATE_PIN(PORTD, 5)));
			while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTD, 5)));
				
// 			while (ioport_pin_is_low(IOPORT_CREATE_PIN(PORTE, 3)));
// 			while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
//			forward = ads1255_read_fast24();
			forward = ads1255_read();
		}
		
		
		if (direction == 1)
			max5443_set_voltage1(dacindex-dacindex_pulse_height);
		else
			max5443_set_voltage1(dacindex+dacindex_pulse_height);

		
		tc_clear_overflow(&TCF0);
		if (direction == 1)
			dacindex += dacindex_step;
		else
			dacindex -= dacindex_step;
		
		while (!tc_is_overflow(&TCF0)){
			autogainswitch();
			if (gain <= POT_GAIN_30M)
				rgain = 0;
			else
				rgain = gain-1;
				
			while (ioport_pin_is_low(IOPORT_CREATE_PIN(PORTD, 5)));
			while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTD, 5)));
				
// 			while (ioport_pin_is_low(IOPORT_CREATE_PIN(PORTE, 3)));
// 			while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));			
//			reverse = ads1255_read_fast24();
			reverse = ads1255_read();
		}		
		
		if (direction == 1)
			max5443_set_voltage1(dacindex+dacindex_pulse_height);
		else
			max5443_set_voltage1(dacindex-dacindex_pulse_height);
		
	}
	
	pot_exp_stop();
	tc_write_clock_source(&TCF0, TC_CLKSEL_OFF_gc);
	tc_disable(&TCF0);
	tc_write_count(&TCF0,0);
	ads1255_standby();
	
	return;
}

void eis_waveform(int16_t offset, int16_t amplitude, uint32_t freq, uint16_t cycle){
	
	cycles = cycle;
	uint16_t dacindex = ceil(offset*(65535/(double)3000)+32767);

	max5443_set_voltage1(dacindex);
		
	tc_enable(&TCF0);
	tc_set_overflow_interrupt_callback(&TCF0, tcf0_ovf_callback_eis);
	tc_enable(&TCE1);
	tc_enable(&TCE0);
	tc_enable(&TCD0);

	tc_set_overflow_interrupt_callback(&TCE0, tce0_ovf_callback_eis);
	tc_set_overflow_interrupt_callback(&TCD0, tcd0_ovf_callback_eis);
	
	if (freq > 10000)
	{
		skip_samples = 300000UL/freq;
	}	
	
	//////
	
	for (int i = 0; i<500; i++)
		eis_data[i]=0;
		
	//////
	
// 	if (freq > 1000)
// 		samples=ceil(400000/freq);
// 	else
// 		samples = 500;
	samples = 500;
	
	uint32_t timer_period;
	uint8_t temp_div;
	uint16_t V[samples];
	eis_ptr = (uint16_t*)&V;
	up = 1;
	
 	for (int i = 0; i < samples; i++)
 	{
 		V[i] = dacindex + floor((cos(2*M_PI*i/(samples-1))*amplitude)*(65536/(double)3000));
// 		printf("%u\n\r", V[i]);
	}
	
	pot_exp_start();
	
	timer_period = round(F_CPU/((freq)/10.*(double)samples));
	printf("timer_period %lu\n\r", timer_period);
	temp_div = ceil(timer_period/65535.);
	printf("temp_div %u\n\r", temp_div);
	EVSYS.CH0MUX = EVSYS_CHMUX_TCF0_OVF_gc; //TCF0: DAC timer, TCE1: cos iterator, TCE0: cycle counter
	EVSYS.CH1MUX = EVSYS_CHMUX_TCE1_OVF_gc;
//	EVSYS.CH2MUX = EVSYS_CHMUX_PORTE_PIN3_gc;
	EVSYS.CH2MUX = EVSYS_CHMUX_PORTD_PIN5_gc;
	
	
	if (temp_div <= 1){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV1_gc);
		printf("DIV1_gc\n\r");
	}
	else if (temp_div == 2){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV2_gc);
		timer_period /= 2;
	}		
	else if (temp_div <= 4){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV4_gc);
		timer_period /= 4;
	}
	else if (temp_div <= 8){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV8_gc);
		timer_period /= 8;
	}
	else if (temp_div <= 64){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV64_gc);
		timer_period /= 64;
	}
	else if (temp_div <= 256){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV256_gc);
		timer_period /= 256;
	}
	else if (temp_div <= 1024){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV1024_gc);
		timer_period /= 1024;
	}
	else{
		printf("ERR: Frequency/ADC rate is too low\n\r");
		return;
	}
	
	tcf0period = (uint16_t)timer_period;
	printf("tcf0max = %lu\n\r", tcf0period*(uint32_t)samples);
	
	tc_write_period(&TCF0, (uint16_t)timer_period);
	tc_write_period(&TCE1, samples-1);
	tc_write_period(&TCE0, cycles);
	tc_write_period(&TCD0, skip_samples);

	ads1255_wakeup();
	ads1255_rdatac();
	TCF0.CNT = 0;
	TCE1.CNT = 0;
	TCE0.CNT = 1;
	TCD0.CNT = 1;
	
	tc_write_clock_source(&TCE1, TC_CLKSEL_EVCH0_gc);
	tc_write_clock_source(&TCE0, TC_CLKSEL_EVCH1_gc);
	tc_write_clock_source(&TCD0, TC_CLKSEL_EVCH2_gc);
	
	if (freq > 10000)
		tc_set_overflow_interrupt_level(&TCD0, (TC_INT_LEVEL_t)TC_OVFINTLVL_MED_gc);
	else
//		PORTE.INTCTRL = PORT_INT1LVL_MED_gc;
		PORTD.INTCTRL = PORT_INT1LVL_MED_gc;

	tc_set_overflow_interrupt_level(&TCF0, (TC_INT_LEVEL_t)TC_OVFINTLVL_LO_gc);
	tc_set_overflow_interrupt_level(&TCE0, (TC_INT_LEVEL_t)TC_OVFINTLVL_HI_gc);
	
	while (up != 0)
	{				
	}	
	
	tc_write_clock_source(&TCF0, TC_CLKSEL_OFF_gc);
	tc_write_clock_source(&TCE1, TC_CLKSEL_OFF_gc);
	tc_write_clock_source(&TCE0, TC_CLKSEL_OFF_gc);
	tc_write_clock_source(&TCD0, TC_CLKSEL_OFF_gc);
	tc_disable(&TCF0);
	TCF0.CNT = 0x0;
	
	ads1255_standby();
	
	for (uint16_t i=0; i<500; i++){
		if (eis_data[i] !=0)
 			printf("%i, %i\n\r", i, eis_data[i]);
	}		 
//	tc_disable(&TCE1);
//	TCE1.CNT = 0x0;
	return;
}

static void tcf0_ovf_callback_eis(void){
		max5443_set_voltage1(*(eis_ptr+TCE1.CNT));
		return;
}

static void tce0_ovf_callback_eis(void){
//	printf("TCC0_OVF\n\r");
	tc_set_overflow_interrupt_level(&TCF0, (TC_INT_LEVEL_t)TC_OVFINTLVL_OFF_gc);
	tc_set_overflow_interrupt_level(&TCD0, (TC_INT_LEVEL_t)TC_OVFINTLVL_OFF_gc);
//	PORTE.INTCTRL = PORT_INT1LVL_OFF_gc;
	PORTD.INTCTRL = PORT_INT1LVL_OFF_gc;
	
	up = 0;
	tc_set_overflow_interrupt_level(&TCE0, (TC_INT_LEVEL_t)TC_OVFINTLVL_OFF_gc);
	return;
}

static void tcd0_ovf_callback_eis(void){
	uint16_t position = TCE1.CNT;
	int16_t data_buff = ads1255_read_fast();
	
	if (eis_data[position] == 0)
		eis_data[position] = data_buff;
	else
		eis_data[position] = (eis_data[position] & data_buff)+((eis_data[position]^data_buff)>>1);
		
	return;
}

void sw_test(uint16_t frequency){
	uint16_t dacindex_start = round((-1400)*(65536/(double)3000))+32768;
	uint16_t dacindex_stop = round(1400*(65536/(double)3000))+32768;
	uint32_t period;
	
	tc_enable(&TCF0);
	
	//calculate time to ADC trigger
	period = round(((1/(double)frequency)*F_CPU));
	uint8_t temp_div = ceil(period/UINT16_MAX);
	
	if (temp_div == 1)
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV1_gc);
	else if (temp_div == 2){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV2_gc);
		period /= 2;
	}		
	else if (temp_div <= 4){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV4_gc);
		period /= 4;
	}
	else if (temp_div <= 8){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV8_gc);
		period /= 8;
	}
	else if (temp_div <= 64){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV64_gc);
		period /= 64;
	}
	else if (temp_div <= 256){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV256_gc);
		period /= 256;
	}
	else if (temp_div <= 1024){
		tc_write_clock_source(&TCF0,TC_CLKSEL_DIV1024_gc);
		period /= 1024;
	}
	else{
		printf("ERR: Frequency/ADC rate is too low\n\r");
		return;
	}
	
	tc_write_period(&TCF0, (uint16_t)period/2);
	pot_exp_start();
	tc_write_count(&TCF0,0);
	
	while (1){
		tc_clear_overflow(&TCF0);
		while (!tc_is_overflow(&TCF0));
		max5443_set_voltage1(dacindex_start);
		tc_clear_overflow(&TCF0);
		while (!tc_is_overflow(&TCF0));
		max5443_set_voltage1(dacindex_stop);
	}
	pot_exp_stop();
	tc_disable(&TCF0);
	tc_write_count(&TCF0,0);
	return;
}

void menu(uint8_t selection){
	int16_t start = 2000;
	int16_t stop = 2000;
	uint16_t slope = 0;
	uint16_t height = 0;
	short unsigned int scans = 0;
	int16_t v1 = 2000;
	int16_t v2 = 2000;
	uint32_t freq = 120000;
	uint16_t periods = 0;
	uint16_t amp_dac[20];
	uint16_t amp_seconds[20];

	char input = 'x';
				
	switch (selection)
	{
		case MENU_OPS:
			printf("Options:\n\r");
			
			do {
				printf("ADS Buffer (Y/N): ");
				scanf("%c", &input);
				printf("%c\n\r", input);
		
				if (input == 'y' || input == 'Y')
					ads_buff = ADS_BUFF_ON;
				else if (input == 'n' || input == 'N')
					ads_buff = ADS_BUFF_OFF;
			}while (!(input == 'y'||input == 'Y'||input == 'n'||input == 'N'));	
			
			do {
				printf("ADS sample rate: \n\r");
				printf("1  2.5 Hz\n\r");
				printf("2  5 Hz\n\r");
				printf("3  10 Hz\n\r");
				printf("4  15 Hz\n\r");
				printf("5  25 Hz\n\r");
				printf("6  30 Hz\n\r");
				printf("7  50 Hz\n\r");
				printf("8  60 Hz\n\r");
				printf("9  100 Hz\n\r");
				printf("10 500 Hz\n\r");
				printf("11 1 kHz\n\r");
				printf("12 2 kHz\n\r");
				printf("13 3.75 kHz\n\r");
				printf("14 7.5 kHz\n\r");
				printf("15 15 kHz\n\r");
				printf("16 30 kHz\n\r");
				
				scanf("%u", &periods);
				printf("%u\n\r", periods);
		
				if (periods == 1)
					ads_datarate = ADS_DR_2_5;
				else if (periods == 2)
					ads_datarate = ADS_DR_5;
				else if (periods == 3)
					ads_datarate = ADS_DR_10;
				else if (periods == 4)
					ads_datarate = ADS_DR_15;
				else if (periods == 5)
					ads_datarate = ADS_DR_25;
				else if (periods == 6)
					ads_datarate = ADS_DR_30;
				else if (periods == 7)
					ads_datarate = ADS_DR_50;
				else if (periods == 8)
					ads_datarate = ADS_DR_60;
				else if (periods == 9)
					ads_datarate = ADS_DR_100;
				else if (periods == 10)
					ads_datarate = ADS_DR_500;
				else if (periods == 11)
					ads_datarate = ADS_DR_1000;
				else if (periods == 12)
					ads_datarate = ADS_DR_2000;
				else if (periods == 13)
					ads_datarate = ADS_DR_3750;
				else if (periods == 14)
					ads_datarate = ADS_DR_7500;
				else if (periods == 15)
					ads_datarate = ADS_DR_15000;
				else if (periods == 16)
					ads_datarate = ADS_DR_30000;
			}while (periods == 0 || periods > 16);
			
			do {
				printf("ADS PGA (1,2,4,8,16,32,64): \n\r");
				
				scanf("%u", &periods);
				printf("%u\n\r", periods);
		
				if (periods == 1)
					ads_pga = ADS_PGA_1;
				else if (periods == 2)
					ads_pga = ADS_PGA_2;
				else if (periods == 4)
					ads_pga = ADS_PGA_4;
				else if (periods == 8)
					ads_pga = ADS_PGA_8;
				else if (periods == 16)
					ads_pga = ADS_PGA_16;
				else if (periods == 32)
					ads_pga = ADS_PGA_32;
				else if (periods == 64)
					ads_pga = ADS_PGA_64;
				else
					periods = 0;
			}while (periods == 0 || periods > 64);
			
			ads1255_setup(ads_buff,ads_datarate,ads_pga);
			
			break;
		case MENU_LSV:
			printf("LSV:\n\r");
				
			do {
				printf("Start (-1500 - 1500 mV): ");
				scanf("%d", &start);
				printf("%d\n\r", start);
		
				if (start > 1500 && start < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (start > 1500 && start < -1500);				
				
			do {
				printf("Stop (-1500 - 1500 mV): ");
				scanf("%d", &stop);
				printf("%d\n\r", stop);
		
				if (stop > 1500 && stop < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (stop > 1500 && stop < -1500);
				
			do {
				printf("Slope (1 - 7000 mV/s): ");
				scanf("%u", &slope);
				printf("%u\n\r", slope);
		
				if (slope > 7000 && slope < 1)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (slope > 7000 && slope < 1);
				
			lsv_experiment(start,stop,slope,2);
			break;
				
		case MENU_CV:
			printf("CV:\n\r");
				
			do {
				printf("v1 (-1500 - 1500 mV): ");
				scanf("%d", &v1);
				printf("%d\n\r", v1);
		
				if (v1 > 1500 && v1 < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (v1 > 1500 && v1 < -1500);				
				
			do {
				printf("v2 (-1500 - 1500 mV): ");
				scanf("%d", &v2);
				printf("%d\n\r", v2);
		
				if (v2 > 1500 && v2 < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (v2 > 1500 && v2 < -1500);
				
			do {
				printf("Start (-1500 - 1500 mV): ");
				scanf("%d", &start);
				printf("%d\n\r", start);
		
				if (start > 1500 && start < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (start > 1500 && start < -1500);	
				
			do {
				printf("Slope (1 - 7000 mV/s): ");
				scanf("%u", &slope);
				printf("%u\n\r", slope);
		
				if (slope > 7000 && slope < 1)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (slope > 7000 && slope < 1);
				
			do {
				printf("Scans (1-255): ");
				scanf("%hu", &scans);
				printf("%hu\n\r", scans);
		
				if (scans < 1)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (scans < 1);	
				
			cv_experiment(v1,v2,start,scans,slope);
			break;
		case MENU_EIS:
			printf("EIS:\n\r");
				
			do {
				printf("freq (1-100000 dHz): \n\r");
				scanf("%lu", &freq);
		
				if (freq > 100000 || freq == 0)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (freq > 100000 || freq == 0);
				
			do {
				printf("offset (-1500-1500 mV): \n\r");
				scanf("%d", &start);
		
				if (start > 1500 || start < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (start > 1500 || start < -1500);
				
			do {
				printf("amplitude (1-500 mV): \n\r");
				scanf("%u", &slope);
		
				if (slope > 500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (slope > 500);
				
			do {
				printf("cycles: \n\r");
				scanf("%u", &periods);
		
				if (periods > 10000 || periods == 0)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (periods > 10000 || periods == 0);		
			ads1255_setup(ADS_BUFF_ON,ADS_DR_30000,ADS_PGA_2);
			eis_waveform(start, slope, freq, periods);
			break;
		case MENU_SWV:
			printf("SWV:\n\r");
				
			do {
				printf("Start (-1500-1500 mV): ");
				scanf("%d", &start);
				printf("%d \n\r",start);
		
				if (start > 1500 || start < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (start > 1500 || start < -1500);
				
			do {
				printf("Stop (-1500-1500 mV): ");
				scanf("%d", &stop);
				printf("%d \n\r", stop);
		
				if (stop > 1500 || stop < -1500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (stop > 1500 || stop < -1500);
				
			do {
				printf("Step (1-400 mV): ");
				scanf("%u", &slope);
				printf("%u \n\r", slope);
		
				if (slope > 400 || slope < 1)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (slope > 400 || slope < 1);
				
			do {
				printf("Pulse height (1-500 mV): ");
				scanf("%u", &height);
				printf("%u \n\r", height);
		
				if (height > 500)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (height > 500);
				
			do {
				printf("freq (1-10000 Hz): ");
				scanf("%lu", &freq);
				printf("%lu\n\r", freq);
		
				if (freq > 10000 || freq == 0)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (freq > 10000 || freq == 0);
			
			do {
				printf("Scans: ");
				scanf("%u", &periods);
				printf("%u\n\r", periods);
		
				if (periods == 0)
					printf("Invalid selection.\n\r");
			}while (periods == 0);
		
			for (uint16_t i = 0; i < periods; ++i)
				swv_experiment(start, stop, slope, height, freq);
			break;
			
		case MENU_AMP:
			printf("Chronoamperometry:\n\r");
			
			do {
				printf("Steps (1-20): ");
				scanf("%u", &slope);
				printf("%u\n\r", slope);
		
				if (slope > 20 || slope == 0)
				{
					printf("Invalid selection.\n\r");
				} 
			}while (slope > 20 || slope == 0);
			
			for (uint16_t i = 0; i<slope; ++i)
			{
				printf("Step %u:\n\r", i);
				
				do {
					printf("Time (s): ");
					scanf("%u", &amp_seconds[i]);
					printf("%u\n\r", amp_seconds[i]);
					
					if (amp_seconds[i] == 0)
						printf("Invalid selection.\n\r");
				}while (amp_seconds[i] == 0);
				
				do {
					printf("Voltage (-1500-1500 mV): ");
					scanf("%d", &start);
					printf("%d\n\r", start);
					if (start > 1500 || start < -1500)
						printf("Invalid selection.\n\r");
				}while (start > 1500 || start < -1500);
				
				amp_dac[i] = ceil(start*(65535/(double)3000)+32767);
			}
			chronoamp(slope, amp_dac, amp_seconds);
			
			break;
		default:
			printf("Invalid selection.\n\r");
			return;
		return;
	}
}	

int main (void)
{
	board_init();
	pmic_init();
	
	irq_initialize_vectors();
	cpu_irq_enable();
	sleepmgr_init();
	sysclk_init(); //Disables ALL peripheral clocks D:
	rtc_init();
	sysclk_enable_module(SYSCLK_PORT_GEN, SYSCLK_EVSYS);
	
	pmic_set_scheduling(PMIC_SCH_ROUND_ROBIN);
	
	stdio_usb_init(&USARTD1);
	stdio_usb_enable();
//	printf("DStat v1 25 Jun 2012\n\r");
	
	ads1255_init_pins();
	ads1255_init_module();
	ads1255_setup(ADS_BUFF_ON,ADS_DR_60,ADS_PGA_2);
// // 	PORTE.INT0MASK = PIN3_bm;
// // 	PORTE.INT1MASK = PIN3_bm;
// // 	PORTE.INTCTRL = PORT_INT0LVL_OFF_gc;
// // 	PORTE.INTCTRL = PORT_INT1LVL_OFF_gc;
	PORTD.INT0MASK = PIN5_bm;
	PORTD.INT1MASK = PIN5_bm;
	PORTD.INTCTRL = PORT_INT0LVL_OFF_gc | PORT_INT1LVL_OFF_gc;
	
	pot_exp_stop();
	max5443_init_pins();
	max5443_init_module();
	delay_ms(300);
	
	uint16_t input;
	
	gain = POT_GAIN_3k;
 	pot_set_gain(gain);
 	
// 	RTC.CTRL = RTC_PRESCALER_OFF_gc;
// 	while (RTC.STATUS & RTC_SYNCBUSY_bm);
// 	RTC.CNT = 0;
// 	RTC.PER = 0xffff;
// 	RTC.CTRL = RTC_PRESCALER_DIV1_gc;
	
	ads1255_wakeup();
	ads1255_rdatac();
	int32_t data1 = 0;

//	max5443_set_voltage1(21845); //1V
//	pot_exp_start();
	
	while (1){		
		ads1255_standby();
	
		printf("Choose Experiment type:\n\r");
		printf("Options:0, LSV:1, CV:2, SWV:3, AMP:4, EIS:5\n\r");
		printf("DStat v1.1 27 Aug 2012\n\r");
		
		scanf("%u", &input);
		
		if (input < 6)
			menu(input);
		else
			printf("Invalid selection.\n\r");
		
		getchar();	
		
	}		
			
}


ISR(PORTD_INT0_vect){
	if (portd_int0_callback) {
		portd_int0_callback();
	}
// 	if (autogainswitch()==0){
// 		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
// 		if (gain == POT_GAIN_30k)
// 		{
// 			printf("%u, %ld\n\r", TCC1.CNT, ads1255_read()*100);
// 		}
// 		else if (gain == POT_GAIN_300k)
// 		{
// 			printf("%u, %ld\n\r", TCC1.CNT, ads1255_read()*10);	
// 		}
// 		else if (gain == POT_GAIN_3M)
// 		{
// 			printf("%u, %ld\n\r", TCC1.CNT, ads1255_read());	
// 		}
// 		else
// 		{
// 			printf("%u, %lu\n\r", TCC1.CNT, ads1255_read());
// 		}
// 		/*printf("%li, %li\n\r", voltage, ads1255_read());*/
// 	}
// 	else{
// 		while (ioport_pin_is_high(IOPORT_CREATE_PIN(PORTE, 3)));
// 		ads1255_read();
// 	}
}

ISR(PORTD_INT1_vect){
//	if (autogainswitch()==0){
//	printf("INT1\n\r");
//  	static uint16_t skip = 0;
//  	if (++skip <= skip_samples)
//  		return;
//  	skip = 0;
// 	
// 	if (samples == 500){
// 			printf("%u, %d\n\r", TCE1.CNT, ads1255_read_fast());
// 			return;
// 		}
// 
// 	uint32_t iter1 = TCE1.CNT*(uint32_t)tcf0period + TCF0.CNT;
// 		
// 	printf("%lu, %d\n\r", iter1, ads1255_read_fast()/**(3./65535/30000.)*/);
	
	uint16_t position = TCE1.CNT;
	int16_t data_buff = ads1255_read_fast();
	
	if (eis_data[position] == 0)
		eis_data[position] = data_buff;
	else
		eis_data[position] = (eis_data[position] & data_buff)+((eis_data[position]^data_buff)>>1);
	
	
	return;

}