//
//  ds4550.c
//  dstat-firmware
//
//  Created by Michael Dryden on 2017-02-18.
//  Copyright © 2017 Michael Dryden. All rights reserved.
//

#include <twi_master.h>
#include <twi_common.h>

#include "ds4550.h"
#include "../config/dstat_config.h"

#define DS4550_TWI  EXT_TWI0

#define DS4550_ADDR 0b1010000

// Memory Addresses
#define EEPROM_USER         0x0
#define EEPROM_USER_SIZE    0x3F
#define PULLUP_0            0xF0
#define PULLUP_1            0xF1
#define IO_CTRL_0           0xF2
#define IO_CTRL_1           0xF3
#define CONFIG              0xF4
#define EEPROM_S_USER       0xF5
#define EEPROM_S_USER_SIZE  0x02
#define IO_STATUS_0         0xF8
#define IO_STATUS_1         0xF9
#define SRAM_USER           0xFA
#define SRAM_USER_SIZE      0x5

static uint8_t read_byte(uint8_t address);
static void read_block(uint8_t address, uint8_t length, uint8_t* buffer);
static void write_byte(uint8_t address, uint8_t value);
static void write_block(uint8_t address, uint8_t length, uint8_t* buffer);

static uint8_t read_byte(uint8_t address) {
    /**
     * Reads one byte from AD5933
     *
     * @param address Address on AD5933
     * @return Value read.
     */
    uint8_t value;
    
    read_block(address, 1, (uint8_t*)&value);
    
    return value;
}

static void read_block(uint8_t address, uint8_t length, uint8_t* buffer) {
    /**
     * Reads a block of data
     *
     * @param address Start address on DS4550
     * @param length number of bytes to read.
     * @param buffer pointer to buffer to store result
     */
    twi_package_t twi_pack;
    
    twi_pack.addr[0] = address;
    twi_pack.addr_length = 1;
    twi_pack.buffer = buffer;
    twi_pack.chip = DS4550_ADDR;
    twi_pack.no_wait = false;
    twi_pack.length = length;
    
    twi_master_read(&DS4550_TWI, &twi_pack);
    return;
}


static void write_byte(uint8_t address, uint8_t value) {
    /**
     * Writes one byte to AD5933.
     *
     * @param address Address on AD5933
     * @param Value to send.
     */
    write_block(address, 1, &value);
}

static void write_block(uint8_t address, uint8_t length, uint8_t* buffer) {
    /**
     * Writes a block of data
     *
     * @param address Start address on DS4550
     * @param length number of bytes to write.
     * @param buffer pointer to buffer to write
     */
    twi_package_t twi_pack;
    
    twi_pack.addr[0] = address;
    twi_pack.addr_length = 1;
    twi_pack.buffer = buffer;
    twi_pack.chip = DS4550_ADDR;
    twi_pack.no_wait = false;
    twi_pack.length = length;
    
    if(twi_master_write(&DS4550_TWI, &twi_pack) != TWI_SUCCESS)
        printf("#ERR: DS4550: TWI write fail\r\n");
}

void ds4550_set_pin(uint8_t pin, uint8_t state) {
    uint16_t current;
    read_block(PULLUP_0, 2, (uint8_t*)&current);
    
    uint16_t output = (1 << pin);
    
    if (!state) {
        current &= ~output;
    }
    else {
        current |= output;
    }
    
    write_block(PULLUP_0, 2, (uint8_t*)&current);
}

void ds4550_set_pins(uint16_t pins, uint16_t pin_mask) {
    uint16_t current;
    read_block(PULLUP_0, 2, (uint8_t*)&current);
    
    pins &= pin_mask;  // Make sure pins doesn't have extra 1s
    current &= ~pin_mask;  // Clear where pin_mask is 1
    
    current |= pins;
    write_block(PULLUP_0, 2, (uint8_t*)&current);
}

uint16_t ds4550_get_pins(void) {
    uint16_t current;
    read_block(PULLUP_0, 2, (uint8_t*)&current);
    
    return current;
}

void ds4550_set_eeprom(uint8_t on) {
    /**
     * Enables or disables persistent pin state.
     *
     * @param on Enables if true, disables if false.
     */
    
    uint8_t current = read_byte(CONFIG);
    
    if (on) {
        write_byte(CONFIG, current & 0b11111110);  // LSB=0 is EEPROM enabled
        printf("#INFO: DS4550: EEPROM enabled\r\n");
    }
    else {
        write_byte(CONFIG, current | 0b00000001);  // LSB=1 is EEPROM disabled
        printf("#INFO: DS4550: EEPROM disabled\r\n");
    }
}


void ds4550_init(void) {
    /**
     * Checks to make sure pin defaults are all 0, then disables EEPROM writes.
     */
    
    if ((ds4550_get_pins() & 0x01FF) != 0) {
        ds4550_set_eeprom(1);
        ds4550_set_pins(0x01FF, 0x01FF);
    }
        
    ds4550_set_eeprom(0);
}
